import { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import { Button } from '@/components/ui/button';
import { Moon, Sun, Menu, X, Zap } from 'lucide-react';

export const Header = () => {
  const [isDark, setIsDark] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const location = useLocation();

  useEffect(() => {
    const theme = localStorage.getItem('theme');
    const systemDark = window.matchMedia('(prefers-color-scheme: dark)').matches;

    if (theme === 'dark' || (!theme && systemDark)) {
      setIsDark(true);
      document.documentElement.classList.add('dark');
    }
  }, []);

  const toggleTheme = () => {
    const newTheme = !isDark;
    setIsDark(newTheme);

    if (newTheme) {
      document.documentElement.classList.add('dark');
      localStorage.setItem('theme', 'dark');
    } else {
      document.documentElement.classList.remove('dark');
      localStorage.setItem('theme', 'light');
    }
  };

  const navItems = [
    { name: 'HOME', href: '/' },
    { name: 'SERVICES', href: '/services' },
    { name: 'PROJECTS', href: '/projects' },
    { name: 'ABOUT', href: '/about' },
    { name: 'CONTACT', href: '/contact' }
  ];

  return (
    <header className="fixed top-0 left-0 right-0 z-50 backdrop-blur-cyber border-b neon-border">
      {/* Scanning line effect */}
      <div className="absolute top-0 left-0 right-0 h-px bg-gradient-to-r from-transparent via-primary to-transparent animate-data-stream"></div>

      <nav className="container mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex items-center justify-between h-20">
          {/* Futuristic Logo */}
          <Link to="/" className="flex items-center group hover:scale-105 transition-transform duration-300 flex-shrink-0">
            <div className="relative">
              <Zap className="w-6 h-6 sm:w-8 sm:h-8 text-primary mr-2 sm:mr-3 group-hover:rotate-12 transition-transform duration-300" />
              <div className="absolute inset-0 w-6 h-6 sm:w-8 sm:h-8 bg-primary/20 rounded-full animate-pulse"></div>
            </div>
            <div className="font-space-age font-steel-blue bg-gradient-neon bg-clip-text">
              <div className="text-lg sm:text-xl lg:text-2xl leading-tight">MATHENOID</div>
              <div className="text-xs sm:text-sm lg:text-base leading-tight">ICT LAB</div>
            </div>
          </Link>
          {/* Desktop & Tablet Navigation - Centered with proper spacing */}
          <div className="hidden md:flex flex-1 justify-center px-8">
            <div className="flex items-baseline space-x-8 lg:space-x-10">
              {navItems.map((item) => (
                <Link
                  key={item.name}
                  to={item.href}
                  className={`relative transition-all duration-300 font-tech font-medium text-sm tracking-wider ${
                    location.pathname === item.href
                      ? 'text-primary text-cyber-glow'
                      : 'text-foreground hover:text-primary group'
                  }`}
                >
                  {item.name}
                  <span className={`absolute bottom-0 left-0 w-full h-0.5 bg-primary transition-transform duration-300 ${
                    location.pathname === item.href ? 'scale-x-100' : 'scale-x-0 group-hover:scale-x-100'
                  }`}></span>
                  {location.pathname === item.href && (
                    <span className="absolute -top-1 -right-1 w-2 h-2 bg-accent rounded-full animate-pulse"></span>
                  )}
                </Link>
              ))}
            </div>
          </div>
          {/* Controls */}
          <div className="flex items-center space-x-3 flex-shrink-0">
            <Button
              variant="ghost"
              size="icon"
              onClick={toggleTheme}
              className="relative border border-primary/30 hover:bg-primary/20 group w-10 h-10"
            >
              {isDark ? (
                <Sun className="h-4 w-4 text-primary group-hover:rotate-180 transition-transform duration-500" />
              ) : (
                <Moon className="h-4 w-4 text-primary group-hover:rotate-180 transition-transform duration-500" />
              )}
              <div className="absolute inset-0 bg-primary/10 rounded-full scale-0 group-hover:scale-100 transition-transform duration-300"></div>
            </Button>

            {/* Mobile cyber menu */}
            <Button
              variant="ghost"
              size="icon"
              className="md:hidden border border-primary/30 hover:bg-primary/20 group w-10 h-10"
              onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
            >
              {isMobileMenuOpen ? (
                <X className="h-4 w-4 text-primary group-hover:rotate-90 transition-transform duration-300" />
              ) : (
                <Menu className="h-4 w-4 text-primary group-hover:scale-110 transition-transform duration-300" />
              )}
            </Button>
          </div>
        </div>
        {/* Modern Mobile Navigation Sidebar */}
        {isMobileMenuOpen && (
          <>
            {/* Backdrop Blur Overlay */}
            <div
              className="md:hidden fixed inset-0 z-[998] bg-black/50 backdrop-blur-md animate-fade-in"
              onClick={() => setIsMobileMenuOpen(false)}
            />

            {/* Mobile Menu Box Container */}
            <div className="md:hidden fixed top-0 left-0 h-full w-80 max-w-[85vw] z-[999] bg-slate-900 shadow-2xl animate-slide-in-right">
              <div className="h-full bg-slate-900">
                {/* Header Section */}
                <div className="flex items-center justify-between p-6 border-b border-primary/10 pb-2">
                  <div className="flex items-center space-x-3">
                    <div className="relative">
                      <Zap className="w-8 h-8 text-primary" />
                      <div className="absolute inset-0 w-8 h-8 bg-primary/20 rounded-full animate-pulse"></div>
                    </div>
                    <div>
                      <h2 className="font-space-age text-lg font-black text-primary">MATHENOID</h2>
                      <p className="text-xs text-muted-foreground">ICT LAB</p>
                    </div>
                  </div>
                </div>
                {/* Navigation Menu */}
                <div className="flex flex-col bg-slate-900 rounded-br-[25px] pt-3">
                  <nav className="space-y-2 px-4">
                    {navItems.map((item, index) => {
                      const isActive = location.pathname === item.href;
                      return (
                        <Link
                          key={item.name}
                          to={item.href}
                          onClick={() => setIsMobileMenuOpen(false)}
                          className={`flex items-center space-x-4 px-4 py-4 rounded-xl transition-all duration-300 group ${
                            isActive
                              ? 'bg-primary/10 text-primary border-l-4 border-primary'
                              : 'text-foreground hover:bg-primary/5 hover:text-primary'
                          }`}
                          style={{
                            animationDelay: `${index * 0.1}s`,
                            animation: 'fade-in 0.6s ease-out forwards',
                            opacity: '0'
                          }}
                        >
                          <div className={`w-6 h-6 rounded-lg flex items-center justify-center ${
                            isActive ? 'bg-primary/20' : 'bg-muted group-hover:bg-primary/10'
                          }`}>
                            <div className={`w-2 h-2 rounded-full ${
                              isActive ? 'bg-primary' : 'bg-muted-foreground group-hover:bg-primary'
                            }`} />
                          </div>
                          <span className="font-tech font-medium text-base tracking-wide">
                            {item.name}
                          </span>
                          {isActive && (
                            <div className="ml-auto w-2 h-2 bg-primary rounded-full animate-pulse" />
                          )}
                        </Link>
                      );
                    })}
                  </nav>
                  {/* Footer Section */}
                  <div className="mt-auto px-4 pt-8 pb-4 border-t border-primary/10">
                    <div className="flex items-center justify-between text-xs text-muted-foreground">
                      <span>mathenoid ict lab</span>
                      <div className="flex space-x-1">
                        <div className="w-1 h-1 bg-primary rounded-full animate-pulse"></div>
                        <div className="w-1 h-1 bg-primary/60 rounded-full animate-pulse" style={{ animationDelay: '0.5s' }}></div>
                        <div className="w-1 h-1 bg-primary/30 rounded-full animate-pulse" style={{ animationDelay: '1s' }}></div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </>
        )}
      </nav>
    </header>
  );
};
