import { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { useNavigate } from 'react-router-dom';

import { Button } from '@/components/ui/button';
import {
  Globe,
  Smartphone,
  Database,
  Code2,
  Settings,
  Wrench,
  X,
  Star,
  Clock,
  Users,
  Zap,
  Target,
  Layers,
  CheckCircle,
  ArrowRight,
  ChevronUp,
  ChevronDown
} from 'lucide-react';

export const ServicesSection = () => {
  const [hoveredCard, setHoveredCard] = useState<number | null>(null);
  const [selectedService, setSelectedService] = useState<number | null>(null);
  const navigate = useNavigate();

  const handleContactClick = () => {
    navigate('/contact');
  };
  const services = [
    {
      icon: Globe,
      title: 'Web Development',
      description: 'Modern, responsive websites and web applications built with cutting-edge technologies.',
      features: ['React + Node.js', 'Responsive Design', 'SEO Optimized', 'Performance Focused'],
      color: 'from-primary to-steel-blue-light',
      detailedInfo: {
        overview: 'Transform your digital presence with cutting-edge web solutions that combine stunning design with powerful functionality. Our web development services leverage the latest technologies to create fast, secure, and scalable applications.',
        technologies: ['React.js & Next.js', 'Node.js & Express', 'QT', 'TypeScript', 'Tailwind CSS', 'MongoDB, MySQL & PostgreSQL'],
        benefits: ['Lightning-fast performance', 'Mobile-first responsive design', 'SEO optimization', 'Advanced security measures', 'Scalable architecture', 'Real-time functionality'],
        process: ['Discovery & Planning', 'UI/UX Design', 'Development & Testing', 'Deployment & Launch', 'Ongoing Maintenance'],
        timeline: '4-12 weeks',
        pricing: 'Starting from KSH 10,000'
      }
    },
    {
      icon: Smartphone,
      title: 'Mobile Applications',
      description: 'Native and cross-platform mobile apps that deliver exceptional user experiences.',
      features: ['React Native', 'QT', 'iOS & Android', 'App Store/ Play Store Deployment'],
      color: 'from-accent to-orange-warm',
      detailedInfo: {
        overview: 'Create powerful mobile experiences that engage users and drive business growth. Our mobile development expertise spans native iOS/Android and cross-platform solutions.',
        technologies: ['React Native', 'QT Framework', 'Firebase', 'Node.js & Express', 'App Store Connect'],
        benefits: ['Cross-platform compatibility', 'Native performance', 'Offline functionality', 'Push notifications', 'In-app purchases', 'App store optimization'],
        process: ['Market Research', 'Prototype Development', 'Native Development', 'Quality Assurance', 'Store Submission'],
        timeline: '8-16 weeks',
        pricing: 'Starting from KSH 25,000'
      }
    },
    {
      icon: Database,
      title: 'Data Analytics',
      description: 'Transform your data into actionable insights with advanced analytics solutions.',
      features: ['Business Intelligence', 'Data Visualization', 'Predictive Analytics', 'Custom Dashboards'],
      color: 'from-secondary to-charcoal',
      detailedInfo: {
        overview: 'Unlock the power of your data with comprehensive analytics solutions. From data collection to visualization, we help you make data-driven decisions that accelerate growth.',
        technologies: ['Python & R', 'Tableau & Power BI', 'Apache Spark', 'TensorFlow', 'SQL Databases', 'Cloud Analytics'],
        benefits: ['Real-time insights', 'Predictive modeling', 'Custom reporting', 'Data automation', 'Performance tracking', 'ROI optimization'],
        process: ['Data Assessment', 'Pipeline Development', 'Analytics Setup', 'Dashboard Creation', 'Training & Support'],
        timeline: '6-10 weeks',
        pricing: 'Starting from KSH 25,000'
      }
    },
    {
      icon: Code2,
      title: 'Software Solutions',
      description: 'Custom software development tailored to your specific business requirements.',
      features: ['Desktop Applications', 'API Development', 'System Integration', 'Legacy Modernization'],
      color: 'from-steel-blue-dark to-primary',
      detailedInfo: {
        overview: 'Build robust software solutions that streamline operations and enhance productivity. Our custom development approach ensures your software perfectly aligns with your business processes.',
        technologies: ['Javascript', 'QT C++', 'REST & GraphQL APIs', 'Cloud Integration'],
        benefits: ['Custom functionality', 'System integration', 'Scalable architecture', 'Enhanced security', 'Process automation', 'Legacy modernization'],
        process: ['Requirements Analysis', 'Architecture Design', 'Development & Integration', 'Testing & Deployment', 'Documentation & Training'],
        timeline: '10-20 weeks',
        pricing: 'Starting from KSH 25,000'
      }
    },
    {
      icon: Settings,
      title: 'Application Maintenance',
      description: 'Reliable ongoing support and maintenance to keep your applications running smoothly.',
      features: ['Bug Fixes', 'Performance Monitoring', 'Security Updates', '24/7 Support'],
      color: 'from-primary to-accent',
      detailedInfo: {
        overview: 'Ensure your applications perform optimally with our comprehensive maintenance services. From bug fixes to security updates, we keep your software running at peak performance.',
        technologies: ['Monitoring Tools', 'Automated Testing', 'Security Scanning', 'Performance Analytics', 'Cloud Monitoring'],
        benefits: ['99.9% uptime guarantee', 'Proactive monitoring', 'Regular updates', 'Security patches', 'Performance optimization', 'Technical support'],
        process: ['Health Assessment', 'Monitoring Setup', 'Regular Maintenance', 'Issue Resolution', 'Performance Reports'],
        timeline: 'Ongoing service',
        pricing: 'Starting from KSH 500/month'
      }
    },
    {
      icon: Wrench,
      title: 'Software Enhancement',
      description: 'Continuous improvement and feature additions to evolve your software solutions.',
      features: ['Feature Development', 'UI/UX Improvements', 'Performance Optimization', 'Technology Upgrades'],
      color: 'from-charcoal to-steel-blue-dark',
      detailedInfo: {
        overview: 'Evolve your software with continuous enhancements that keep you ahead of the competition. We add new features, improve performance, and modernize your technology stack.',
        technologies: ['Modern Frameworks', 'UI/UX Design Tools', 'Performance Testing', 'Code Refactoring', 'API Upgrades', 'Cloud Migration'],
        benefits: ['Enhanced user experience', 'Improved performance', 'New feature additions', 'Technology modernization', 'Better scalability', 'Competitive advantage'],
        process: ['Feature Planning', 'Design & Development', 'Testing & Integration', 'Deployment & Rollout', 'User Training'],
        timeline: '2-8 weeks per enhancement',
        pricing: 'Starting from KSH 10,000'
      }
    }
  ];

  const Modal = ({ serviceIndex, onClose }) => {
    const service = services[serviceIndex];
    const Icon = service.icon;

    const scrollToTop = () => {
      const scrollContainer = document.querySelector('.modal-scroll-container');
      if (scrollContainer) {
        scrollContainer.scrollTo({ top: 0, behavior: 'smooth' });
      }
    };

    const scrollToBottom = () => {
      const scrollContainer = document.querySelector('.modal-scroll-container');
      if (scrollContainer) {
        scrollContainer.scrollTo({ top: scrollContainer.scrollHeight, behavior: 'smooth' });
      }
    };

    return (
      <div className="fixed border-none inset-0 z-50 flex items-center justify-center rounded-3xl p-4 bg-black/60 backdrop-blur-sm animate-fade-in">
        <div className="relative w-full max-w-4xl max-h-[90vh] bg-background rounded-3xl shadow-2xl overflow-hidden animate-slide-in">
          {/* Close Button */}
          <Button
            variant="ghost"
            size="sm"
            onClick={onClose}
            className="absolute top-2 right-4 z-10 rounded-full w-7 h-7 p-0 bg-background/80 hover:bg-primary/20 animate-pulse"
          >
            <X className="w-5 h-5" />
          </Button>

          {/* Small Scroll Navigation Buttons */}
          <div className="absolute right-4 top-1/2 transform -translate-y-1/2 z-10 flex flex-col gap-2">
            <Button
              variant="ghost"
              size="sm"
              onClick={scrollToTop}
              className="rounded-full w-8 h-8 p-0 bg-background/80 hover:bg-primary/20 shadow-lg"
            >
              <ChevronUp className="w-4 h-4" />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={scrollToBottom}
              className="rounded-full w-8 h-8 p-0 bg-background/80 hover:bg-primary/20 shadow-lg"
            >
              <ChevronDown className="w-4 h-4" />
            </Button>
          </div>

          {/* Scrollable Content Container - Hidden scrollbar */}
          <div className="modal-scroll-container overflow-y-auto max-h-[90vh] scrollbar-hide" style={{
            scrollbarWidth: 'none', /* Firefox */
            msOverflowStyle: 'none', /* Internet Explorer 10+ */
          }}>
            <style jsx>{`
              .scrollbar-hide::-webkit-scrollbar {
                display: none; /* Safari and Chrome */
              }
            `}</style>

            {/* Header */}
            <div className={`sticky px-6 sm:px-8 pt-8 pb-6 bg-gradient-to-br ${service.color} rounded-t-3xl`}>
              <div className="flex items-center gap-4 mb-4">
                <div className="w-16 h-16 bg-white/20 rounded-2xl p-4 backdrop-blur-sm">
                  <Icon className="w-full h-full text-white" />
                </div>
                <div>
                  <h2 className="text-2xl sm:text-3xl font-bold text-white mb-2">{service.title}</h2>
                  <Badge variant="secondary" className="bg-white/20 text-white border-white/30">
                    Professional Service
                  </Badge>
                </div>
              </div>
              <p className="text-white/90 text-lg leading-relaxed">{service.detailedInfo.overview}</p>
            </div>

            {/* Content */}
            <div className="px-6 sm:px-8 py-8">
              {/* Quick Info Cards */}
              <div className="grid grid-cols-1 sm:grid-cols-3 gap-4 mb-8 animate-fade-in">
                <Card className="border-primary/20 bg-gradient-to-br from-primary/5 to-transparent transform transition duration-500 hover:scale-105">
                  <CardContent className="p-4 text-center">
                    <Clock className="w-8 h-8 text-primary mx-auto mb-2" />
                    <p className="text-sm text-muted-foreground mb-1">Timeline</p>
                    <p className="font-semibold">{service.detailedInfo.timeline}</p>
                  </CardContent>
                </Card>
                <Card className="border-accent/20 bg-gradient-to-br from-accent/5 to-transparent transform transition duration-500 hover:scale-105">
                  <CardContent className="p-4 text-center">
                    <Target className="w-8 h-8 text-accent mx-auto mb-2" />
                    <p className="text-sm text-muted-foreground mb-1">Starting Price</p>
                    <p className="font-semibold">{service.detailedInfo.pricing}</p>
                  </CardContent>
                </Card>
                <Card className="border-secondary/20 bg-gradient-to-br from-secondary/5 to-transparent transform transition duration-500 hover:scale-105">
                  <CardContent className="p-4 text-center">
                    <Star className="w-8 h-8 text-secondary mx-auto mb-2" />
                    <p className="text-sm text-muted-foreground mb-1">Quality</p>
                    <p className="font-semibold">Premium</p>
                  </CardContent>
                </Card>
              </div>

              {/* Technologies and Benefits */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 animate-slide-in">
                <div>
                  <h3 className="text-xl font-bold mb-4 flex items-center gap-2">
                    <Layers className="w-5 h-5 text-primary" />
                    Technologies & Tools
                  </h3>
                  <div className="grid grid-cols-2 gap-2">
                    {service.detailedInfo.technologies.map((tech, index) => (
                      <div key={index} className="flex items-center gap-2 p-2 rounded-lg border border-border/50 bg-muted/30 transform transition duration-300 hover:scale-105">
                        <Zap className="w-4 h-4 text-primary" />
                        <span className="text-sm">{tech}</span>
                      </div>
                    ))}
                  </div>
                </div>
                <div>
                  <h3 className="text-xl font-bold mb-4 flex items-center gap-2">
                    <CheckCircle className="w-5 h-5 text-primary" />
                    Key Benefits
                  </h3>
                  <div className="space-y-2">
                    {service.detailedInfo.benefits.map((benefit, index) => (
                      <div key={index} className="flex items-center gap-2 transform transition duration-300 hover:translate-x-2">
                        <div className="w-2 h-2 bg-primary rounded-full" />
                        <span className="text-sm">{benefit}</span>
                      </div>
                    ))}
                  </div>
                </div>
              </div>

              {/* Process */}
              <div className="mt-8 animate-fade-in">
                <h3 className="text-xl font-bold mb-4 flex items-center gap-2">
                  <Users className="w-5 h-5 text-primary" />
                  Development Process
                </h3>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-4">
                  {service.detailedInfo.process.map((step, index) => (
                    <div key={index} className="text-center transform transition duration-300 hover:scale-110">
                      <div className={`w-12 h-12 rounded-full bg-gradient-to-br ${service.color} text-white font-bold flex items-center justify-center mx-auto mb-2`}>
                        {index + 1}
                      </div>
                      <p className="text-sm font-medium">{step}</p>
                    </div>
                  ))}
                </div>
              </div>

              {/* CTA */}
              <div className="mt-8 p-6 rounded-2xl bg-gradient-to-r from-primary/10 to-accent/10 border border-primary/20 animate-pulse">
                <div className="text-center">
                  <h4 className="text-lg font-bold mb-2">Ready to Get Started?</h4>
                  <p className="text-muted-foreground mb-4">Let's discuss your project and create something amazing together.</p>
                  <div className="flex flex-col sm:flex-row gap-3 justify-center">
                    <Button size="lg" onClick={handleContactClick}
                      className="bg-primary hover:bg-primary/90 transform transition duration-300 hover:scale-105">
                      Start Your Project
                      <ArrowRight className="ml-2 w-4 h-4" />
                    </Button>
                    <Button variant="outline" onClick={handleContactClick}
                      size="lg" className="transform transition duration-300 hover:scale-105">
                      Schedule Consultation
                    </Button>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  };

  return (
    <section id="services" className="py-20 bg-background">
      <div className="container mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section Header */}
        <div className="text-center mb-16">
          <Badge variant="outline" className="mb-4 border-primary text-primary">
            Our Services
          </Badge>
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold mb-6">
            Comprehensive
            <span className="block gradient-hero bg-clip-text text-transparent">
              ICT Solutions
            </span>
          </h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            From concept to deployment, we provide end-to-end technology solutions
            that drive innovation and accelerate your business growth.
          </p>
        </div>
        {/* Services Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {services.map((service, index) => {
            const Icon = service.icon;
            return (
              <Card
                key={index}
                className={`group cursor-pointer transition-all duration-300 hover:shadow-xl hover:-translate-y-2 border-border hover:border-primary/50 ${hoveredCard === index ? 'scale-105' : ''
                  }`}
                onMouseEnter={() => setHoveredCard(index)}
                onMouseLeave={() => setHoveredCard(null)}
              >
                <CardHeader className="text-center pb-4">
                  <div className={`w-16 h-16 mx-auto rounded-2xl bg-gradient-to-br ${service.color} p-4 mb-4 group-hover:scale-110 transition-transform duration-300`}>
                    <Icon className="w-full h-full text-white" />
                  </div>
                  <CardTitle className="text-xl mb-2 group-hover:text-primary transition-colors">
                    {service.title}
                  </CardTitle>
                  <CardDescription className="text-sm">
                    {service.description}
                  </CardDescription>
                </CardHeader>

                <CardContent>
                  {/* Features List */}
                  <div className="space-y-2 mb-6">
                    {service.features.map((feature, featureIndex) => (
                      <div key={featureIndex} className="flex items-center gap-2">
                        <CheckCircle className="w-4 h-4 text-primary flex-shrink-0" />
                        <span className="text-sm text-muted-foreground">{feature}</span>
                      </div>
                    ))}
                  </div>

                  {/* Learn More Button */}
                  <Button
                    variant="ghost"
                    className="w-full group-hover:bg-primary group-hover:text-primary-foreground transition-colors"
                    onClick={() => setSelectedService(index)}
                  >
                    Learn More
                    <ArrowRight className="ml-2 w-4 h-4 group-hover:translate-x-1 transition-transform" />
                  </Button>
                </CardContent>
              </Card>
            );
          })}
        </div>
        {/* Call to Action */}
        <div className="text-center mt-16">
          <Card className="max-w-2xl mx-auto border-primary/20 bg-gradient-to-r from-primary/5 to-accent/5">
            <CardContent className="pt-8">
              <h3 className="text-2xl font-bold mb-4">Ready to Start Your Project?</h3>
              <p className="text-muted-foreground mb-6">
                Let's discuss how we can transform your ideas into powerful digital solutions.
              </p>
              <Button size="lg" onClick={handleContactClick} className="bg-primary hover:bg-primary/90">
                Get Free Consultation
                <ArrowRight className="ml-2 w-4 h-4" />
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
      {/* Modal */}
      {selectedService !== null && (
        <Modal
          serviceIndex={selectedService}
          onClose={() => setSelectedService(null)}
        />
      )}
    </section>
  );
};