import React, { useState, useEffect } from "react";
import { Header } from "@/components/Header";
import { Footer } from "@/components/Footer";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Alert, AlertDescription } from "@/components/ui/alert";
import {
  Mail,
  Phone,
  MapPin,
  Clock,
  Send,
  MessageCircle,
  Calendar,
  Zap,
  Globe,
  Linkedin,
  Github,
  CheckCircle,
  AlertCircle,
  Loader2,
} from "lucide-react";

// Type definitions
interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  service: string;
  message: string;
  budget: string;
}

interface FormErrors {
  firstName?: string;
  lastName?: string;
  email?: string;
  phone?: string;
  message?: string;
}

type SubmitStatus = "success" | "error" | null;

const Contact = () => {
  // Form state
  const [formData, setFormData] = useState<FormData>({
    firstName: "",
    lastName: "",
    email: "",
    phone: "",
    service: "",
    message: "",
    budget: "",
  });

  // UI state
  const [isSubmitting, setIsSubmitting] = useState<boolean>(false);
  const [submitStatus, setSubmitStatus] = useState<SubmitStatus>(null);
  const [errors, setErrors] = useState<FormErrors>({});

  // Auto-hide success message after 5 seconds
  useEffect(() => {
    if (submitStatus === "success") {
      const timer = setTimeout(() => {
        setSubmitStatus(null);
      }, 5000);

      return () => clearTimeout(timer);
    }
  }, [submitStatus]);

  const contactInfo = [
    {
      icon: Mail,
      title: "Email Us",
      value: "admin@mathenoid.com",
      description: "Get in touch for project inquiries",
      color: "from-blue-500 to-cyan-500",
    },
    {
      icon: Phone,
      title: "Call Us",
      value: "+254 740 46 7735",
      description: "Direct line for urgent matters",
      color: "from-green-500 to-emerald-500",
    },
    {
      icon: MapPin,
      title: "Visit Us",
      value: "JUJA, NRB",
      description: "Available for in-person meetings",
      color: "from-purple-500 to-indigo-500",
    },
    {
      icon: Clock,
      title: "Working Hours",
      value: "24/7 Support",
      description: "We're here when you need us",
      color: "from-orange-500 to-red-500",
    },
  ];

  const services = [
    "Custom Software Development",
    "Mobile App Development",
    "Web Development",
    "AI & Machine Learning",
    "UI/UX Design",
    "Software Enhancement",
    "Software Maintenance",
    "Consultation",
  ];

  const socialLinks = [
    {
      icon: Linkedin,
      href: "https://www.linkedin.com/in/kelvin-nyagah-07b369255",
      label: "LinkedIn",
    },
    {
      icon: Github,
      href: "https://www.github.com/mathenoidke",
      label: "GitHub",
    },
  ];

  // Handle input changes - Fixed TypeScript typing
  const handleInputChange = (
    e: React.ChangeEvent<
      HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement
    >
  ) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value,
    }));

    // Clear error for this field when user starts typing
    if (errors[name as keyof FormErrors]) {
      setErrors((prev) => ({
        ...prev,
        [name]: "",
      }));
    }
  };

  // Validate form
  const validateForm = (): FormErrors => {
    const newErrors: FormErrors = {};

    if (!formData.firstName.trim())
      newErrors.firstName = "First name is required";
    if (!formData.lastName.trim()) newErrors.lastName = "Last name is required";
    if (!formData.email.trim()) newErrors.email = "Email is required";
    if (!formData.message.trim()) newErrors.message = "Message is required";

    // Email format validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (formData.email && !emailRegex.test(formData.email)) {
      newErrors.email = "Please enter a valid email address";
    }

    // Phone validation (if provided)
    if (formData.phone && formData.phone.length < 10) {
      newErrors.phone = "Please enter a valid phone number";
    }

    return newErrors;
  };

  // Handle form submission
  const handleSubmit = async () => {
    // Validate form
    const formErrors = validateForm();
    if (Object.keys(formErrors).length > 0) {
      setErrors(formErrors);
      return;
    }

    setIsSubmitting(true);
    setSubmitStatus(null);
    setErrors({});

    try {
      const response = await fetch("https://website.mathenoid.com/api/contact", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      });

      const data = await response.json();

      if (data.success) {
        setSubmitStatus("success");
        // Reset form
        setFormData({
          firstName: "",
          lastName: "",
          email: "",
          phone: "",
          service: "",
          message: "",
          budget: "",
        });
      } else {
        setSubmitStatus("error");
        if (data.errors && Array.isArray(data.errors)) {
          // Handle validation errors from backend
          const backendErrors: FormErrors = {};
          data.errors.forEach((error: string) => {
            if (error.includes("firstName")) backendErrors.firstName = error;
            if (error.includes("lastName")) backendErrors.lastName = error;
            if (error.includes("email")) backendErrors.email = error;
            if (error.includes("phone")) backendErrors.phone = error;
            if (error.includes("message")) backendErrors.message = error;
          });
          setErrors(backendErrors);
        }
      }
    } catch (error) {
      console.error("Form submission error:", error);
      setSubmitStatus("error");
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="min-h-screen">
      <Header />

      {/* Hero Section */}
      <section className="pt-32 pb-16 px-4 sm:px-6 lg:px-8">
        <div className="container mx-auto text-center">
          <div className="max-w-4xl mx-auto">
            <Badge
              variant="outline"
              className="mb-6 text-primary border-primary/20 font-tech"
            >
              Get In Touch
            </Badge>
            <h1 className="font-space-age text-5xl md:text-7xl font-bold mb-6 bg-gradient-to-r from-primary via-purple-500 to-steel-blue-light bg-clip-text text-transparent animate-hologram">
              Let's Build Something Amazing
            </h1>
            <p className="text-xl text-muted-foreground mb-8 leading-relaxed font-cyber">
              Ready to transform your ideas into reality? We're here to listen,
              advise, and create solutions that exceed your expectations.
            </p>
          </div>
        </div>
      </section>

      {/* Contact Info Cards */}
      <section className="py-16 px-4 sm:px-6 lg:px-8">
        <div className="container mx-auto">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8 mb-16">
            {contactInfo.map((info, index) => {
              const Icon = info.icon;
              return (
                <Card
                  key={index}
                  className="p-6 text-center border-0 hover:shadow-2xl transition-all duration-500 bg-gradient-to-b from-background/80 to-muted/20 group hover:scale-105"
                >
                  <div
                    className={`w-16 h-16 rounded-2xl bg-gradient-to-r ${info.color} p-4 mx-auto mb-4 group-hover:rotate-12 transition-transform duration-300`}
                  >
                    <Icon className="w-full h-full text-white" />
                  </div>
                  <h3 className="text-lg font-bold mb-2 text-foreground font-tech">
                    {info.title}
                  </h3>
                  <p className="text-primary font-semibold mb-2 font-space-age">
                    {info.value}
                  </p>
                  <p className="text-sm text-muted-foreground font-cyber">
                    {info.description}
                  </p>
                </Card>
              );
            })}
          </div>
        </div>
      </section>

      {/* Main Contact Section */}
      <section className="py-16 px-4 sm:px-6 lg:px-8 bg-gradient-to-b from-background to-muted/20">
        <div className="container mx-auto">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-16">
            {/* Contact Form */}
            <Card className="p-8 border-0 shadow-2xl bg-background/80 backdrop-blur-sm">
              <div className="mb-8">
                <h2 className="text-3xl font-bold mb-4 text-foreground font-space-age">
                  Start Your Project
                </h2>
                <p className="text-muted-foreground font-cyber">
                  Fill out the form below and we'll get back to you within 24
                  hours with a detailed proposal.
                </p>
              </div>

              {/* Success/Error Messages */}
              {submitStatus === "success" && (
                <Alert className="mb-6 border-green-200 bg-green-50">
                  <CheckCircle className="h-4 w-4 text-green-600" />
                  <AlertDescription className="text-green-800">
                    Thank you for your message! We'll get back to you within 2
                    hours during business hours.
                  </AlertDescription>
                </Alert>
              )}

              {submitStatus === "error" && (
                <Alert className="mb-6 border-red-200 bg-red-50">
                  <AlertCircle className="h-4 w-4 text-red-600" />
                  <AlertDescription className="text-red-800">
                    Something went wrong. Please try again or contact us
                    directly at admin@mathenoid.com
                  </AlertDescription>
                </Alert>
              )}

              <div className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <label className="text-sm font-medium text-foreground mb-2 block">
                      First Name *
                    </label>
                    <Input
                      name="firstName"
                      value={formData.firstName}
                      onChange={handleInputChange}
                      placeholder="Msee"
                      className={`bg-background border-muted ${
                        errors.firstName ? "border-red-500" : ""
                      }`}
                      disabled={isSubmitting}
                    />
                    {errors.firstName && (
                      <p className="text-red-500 text-sm mt-1">
                        {errors.firstName}
                      </p>
                    )}
                  </div>
                  <div>
                    <label className="text-sm font-medium text-foreground mb-2 block">
                      Last Name *
                    </label>
                    <Input
                      name="lastName"
                      value={formData.lastName}
                      onChange={handleInputChange}
                      placeholder="Mkenya"
                      className={`bg-background border-muted ${
                        errors.lastName ? "border-red-500" : ""
                      }`}
                      disabled={isSubmitting}
                    />
                    {errors.lastName && (
                      <p className="text-red-500 text-sm mt-1">
                        {errors.lastName}
                      </p>
                    )}
                  </div>
                </div>

                <div>
                  <label className="text-sm font-medium text-foreground mb-2 block">
                    Email Address *
                  </label>
                  <Input
                    type="email"
                    name="email"
                    value={formData.email}
                    onChange={handleInputChange}
                    placeholder="mkenya@example.com"
                    className={`bg-background border-muted ${
                      errors.email ? "border-red-500" : ""
                    }`}
                    disabled={isSubmitting}
                  />
                  {errors.email && (
                    <p className="text-red-500 text-sm mt-1">{errors.email}</p>
                  )}
                </div>

                <div>
                  <label className="text-sm font-medium text-foreground mb-2 block">
                    Phone Number
                  </label>
                  <Input
                    type="tel"
                    name="phone"
                    value={formData.phone}
                    onChange={handleInputChange}
                    placeholder="+254 01 23 4567"
                    className={`bg-background border-muted ${
                      errors.phone ? "border-red-500" : ""
                    }`}
                    disabled={isSubmitting}
                  />
                  {errors.phone && (
                    <p className="text-red-500 text-sm mt-1">{errors.phone}</p>
                  )}
                </div>

                <div>
                  <label className="text-sm font-medium text-foreground mb-2 block">
                    Service Needed
                  </label>
                  <select
                    name="service"
                    value={formData.service}
                    onChange={handleInputChange}
                    className="w-full p-3 rounded-md border border-muted bg-background text-foreground"
                    disabled={isSubmitting}
                  >
                    <option value="">Select a service</option>
                    {services.map((service, idx) => (
                      <option key={idx} value={service}>
                        {service}
                      </option>
                    ))}
                  </select>
                </div>

                <div>
                  <label className="text-sm font-medium text-foreground mb-2 block">
                    Project Details *
                  </label>
                  <Textarea
                    name="message"
                    value={formData.message}
                    onChange={handleInputChange}
                    placeholder="Tell us about your project, goals, timeline, and any specific requirements..."
                    className={`min-h-32 bg-background border-muted ${
                      errors.message ? "border-red-500" : ""
                    }`}
                    disabled={isSubmitting}
                  />
                  {errors.message && (
                    <p className="text-red-500 text-sm mt-1">
                      {errors.message}
                    </p>
                  )}
                </div>

                <div>
                  <label className="text-sm font-medium text-foreground mb-2 block">
                    Budget Range
                  </label>
                  <select
                    name="budget"
                    value={formData.budget}
                    onChange={handleInputChange}
                    className="w-full p-3 rounded-md border border-muted bg-background text-foreground"
                    disabled={isSubmitting}
                  >
                    <option value="">Select budget range</option>
                    <option value="5k-20k">5,000 - 20,000</option>
                    <option value="20k-100k">20,000 - 100,000</option>
                    <option value="100k-150k">100,000 - 500,000</option>
                    <option value="500k+">500,000+</option>
                  </select>
                </div>

                <Button
                  type="button"
                  onClick={handleSubmit}
                  size="lg"
                  className="w-full bg-gradient-to-r from-primary to-steel-blue-light hover:opacity-90 text-white"
                  disabled={isSubmitting}
                >
                  {isSubmitting ? (
                    <>
                      <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                      Sending Message...
                    </>
                  ) : (
                    <>
                      <Send className="w-5 h-5 mr-2" />
                      Send Message
                    </>
                  )}
                </Button>
              </div>
            </Card>

            {/* Additional Info */}
            <div className="space-y-8">
              {/* Quick Contact */}
              <Card className="p-8 border-0 bg-gradient-to-br from-primary/10 to-steel-blue-light/10">
                <h3 className="text-2xl font-bold mb-6 text-foreground">
                  Quick Contact
                </h3>
                <div className="space-y-4">
                  <Button
                    variant="outline"
                    className="w-full justify-start text-left h-auto p-4"
                  >
                    <MessageCircle className="w-5 h-5 mr-3 text-primary" />
                    <div>
                      <div className="font-semibold">Live Chat</div>
                      <div className="text-sm text-muted-foreground">
                        Available 24/7
                      </div>
                    </div>
                  </Button>
                  <Button
                    variant="outline"
                    className="w-full justify-start text-left h-auto p-4"
                  >
                    <Calendar className="w-5 h-5 mr-3 text-primary" />
                    <div>
                      <div className="font-semibold">Schedule a Call</div>
                      <div className="text-sm text-muted-foreground">
                        Book a free consultation
                      </div>
                    </div>
                  </Button>
                  <Button
                    variant="outline"
                    className="w-full justify-start text-left h-auto p-4"
                  >
                    <Zap className="w-5 h-5 mr-3 text-primary" />
                    <div>
                      <div className="font-semibold">Emergency Support</div>
                      <div className="text-sm text-muted-foreground">
                        For urgent issues
                      </div>
                    </div>
                  </Button>
                </div>
              </Card>

              {/* What to Expect */}
              <Card className="p-8 border-0 bg-background/80 backdrop-blur-sm">
                <h3 className="text-2xl font-bold mb-6 text-foreground">
                  What to Expect
                </h3>
                <div className="space-y-4">
                  <div className="flex items-start">
                    <div className="w-8 h-8 rounded-full bg-gradient-to-r from-primary to-steel-blue-light flex items-center justify-center text-white text-sm font-bold mr-4 mt-1">
                      1
                    </div>
                    <div>
                      <h4 className="font-semibold text-foreground mb-1">
                        Initial Response
                      </h4>
                      <p className="text-sm text-muted-foreground">
                        We'll respond within 2 hours during business hours
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start">
                    <div className="w-8 h-8 rounded-full bg-gradient-to-r from-primary to-steel-blue-light flex items-center justify-center text-white text-sm font-bold mr-4 mt-1">
                      2
                    </div>
                    <div>
                      <h4 className="font-semibold text-foreground mb-1">
                        Discovery Call
                      </h4>
                      <p className="text-sm text-muted-foreground">
                        30-minute consultation to understand your needs
                      </p>
                    </div>
                  </div>
                  <div className="flex items-start">
                    <div className="w-8 h-8 rounded-full bg-gradient-to-r from-primary to-steel-blue-light flex items-center justify-center text-white text-sm font-bold mr-4 mt-1">
                      3
                    </div>
                    <div>
                      <h4 className="font-semibold text-foreground mb-1">
                        Detailed Proposal
                      </h4>
                      <p className="text-sm text-muted-foreground">
                        Comprehensive plan with timeline and pricing
                      </p>
                    </div>
                  </div>
                </div>
              </Card>

              {/* Social Links */}
              <Card className="p-8 border-0 bg-gradient-to-br from-background/80 to-muted/20">
                <h3 className="text-2xl font-bold mb-6 text-foreground">
                  Connect With Us
                </h3>
                <div className="flex space-x-4">
                  {socialLinks.map((social, index) => {
                    const Icon = social.icon;
                    const link = social.href;
                    return (
                      <Button
                        key={index}
                        variant="outline"
                        size="icon"
                        className="hover:bg-primary/10"
                        onClick={() => window.open(link, "_blank")}
                      >
                        <Icon className="w-5 h-5" />
                      </Button>
                    );
                  })}
                </div>
                <p className="text-sm text-muted-foreground mt-4">
                  Follow us for the latest updates, tech insights, and
                  behind-the-scenes content.
                </p>
              </Card>
            </div>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default Contact;
