import React, { useState, useRef, useEffect } from 'react';
import { Header } from '@/components/Header';
import { Footer } from '@/components/Footer';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { useNavigate } from 'react-router-dom';

import { 
  ExternalLink, 
  Github, 
  Eye,
  Users,
  Award,
  TrendingUp,
  Sparkles,
  MessageCircle,
  X,
  Play,
  Pause,
  Volume2,
  VolumeX,
  Maximize,
  RotateCcw
} from 'lucide-react';

// Creative Video Modal Component
const VideoModal = ({ isOpen, onClose, videoUrl, title, category }) => {
  const [isPlaying, setIsPlaying] = useState(false);
  const [isMuted, setIsMuted] = useState(false);
  const [progress, setProgress] = useState(0);
  const [duration, setDuration] = useState(0);
  const [currentTime, setCurrentTime] = useState(0);
  const videoRef = useRef(null);

  useEffect(() => {
    if (isOpen && videoRef.current) {
      videoRef.current.currentTime = 0;
    }
  }, [isOpen]);

  const togglePlay = () => {
    if (videoRef.current) {
      if (isPlaying) {
        videoRef.current.pause();
      } else {
        videoRef.current.play();
      }
      setIsPlaying(!isPlaying);
    }
  };

  const toggleMute = () => {
    if (videoRef.current) {
      videoRef.current.muted = !isMuted;
      setIsMuted(!isMuted);
    }
  };

  const handleTimeUpdate = () => {
    if (videoRef.current) {
      const current = videoRef.current.currentTime;
      const total = videoRef.current.duration;
      setCurrentTime(current);
      setProgress((current / total) * 100);
    }
  };

  const handleLoadedMetadata = () => {
    if (videoRef.current) {
      setDuration(videoRef.current.duration);
    }
  };

  const handleProgressClick = (e) => {
    if (videoRef.current) {
      const rect = e.currentTarget.getBoundingClientRect();
      const pos = (e.clientX - rect.left) / rect.width;
      videoRef.current.currentTime = pos * duration;
    }
  };

  const formatTime = (time) => {
    const minutes = Math.floor(time / 60);
    const seconds = Math.floor(time % 60);
    return `${minutes}:${seconds.toString().padStart(2, '0')}`;
  };

  const restart = () => {
    if (videoRef.current) {
      videoRef.current.currentTime = 0;
      videoRef.current.play();
      setIsPlaying(true);
    }
  };

    const navigate = useNavigate();

  const handleContactClick = () => {
    navigate('/contact');
  };

  const fullscreen = () => {
    if (videoRef.current) {
      if (videoRef.current.requestFullscreen) {
        videoRef.current.requestFullscreen();
      }
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      {/* Animated Background Overlay */}
      <div 
        className="absolute inset-0 backdrop-blur-sm"
        style={{
          background: 'radial-gradient(circle at center, rgba(0, 0, 0, 0.1) 0%, rgba(0, 0, 0, 0.3) 50%, rgba(0, 0, 0, 0.4) 100%)',
        }}
        onClick={onClose}
      >
        <div className="absolute inset-0 opacity-20">
          <div className="absolute top-1/4 left-1/4 w-3 h-3 bg-steel-blue-light rounded-full animate-pulse shadow-lg shadow-steel-blue-light/50"></div>
          <div className="absolute top-3/4 right-1/3 w-2 h-2 bg-steel-blue-light/80 rounded-full animate-ping shadow-lg shadow-steel-blue-light/30"></div>
          <div className="absolute bottom-1/4 left-1/2 w-2.5 h-2.5 bg-steel-blue-light/60 rounded-full animate-bounce shadow-lg shadow-steel-blue-light/40"></div>
          <div className="absolute top-1/2 right-1/4 w-1.5 h-1.5 bg-steel-blue-light/90 rounded-full animate-pulse shadow-lg shadow-steel-blue-light/50"></div>
          <div className="absolute top-1/3 left-1/3 w-1 h-1 bg-steel-blue-light rounded-full animate-ping shadow-lg shadow-steel-blue-light/60"></div>
          <div className="absolute bottom-1/3 right-1/2 w-2 h-2 bg-steel-blue-light/70 rounded-full animate-bounce shadow-lg shadow-steel-blue-light/40"></div>
        </div>
      </div>

      {/* Modal Container */}
      <div className="relative w-[95vw] max-w-6xl mx-4 animate-in fade-in zoom-in duration-500">
        {/* Main Modal */}
        <div className="relative bg-gradient-to-br from-steel-blue-light/20 via-steel-blue-light/30 to-steel-blue-light/40 backdrop-blur-3xl rounded-3xl overflow-hidden shadow-2xl shadow-steel-blue-light/20 border border-steel-blue-light/20">
          {/* Header */}
          <div className="relative p-6 bg-gradient-to-r from-steel-blue-light/40 via-steel-blue-light/30 to-transparent backdrop-blur-sm">
            <div className="flex items-center justify-between">
              <div>
                <Badge className="mb-2 bg-gradient-to-r from-steel-blue-light to-steel-blue text-white border-0 shadow-lg shadow-steel-blue-light/30">
                  {category}
                </Badge>
                <h3 className="text-2xl font-bold text-white drop-shadow-lg">{title}</h3>
              </div>
              <Button
                variant="ghost"
                size="sm"
                onClick={onClose}
                className="text-white hover:bg-steel-blue-light/30 rounded-full p-2 backdrop-blur-sm border border-steel-blue-light/20 shadow-lg hover:shadow-steel-blue-light/30 transition-all duration-300"
              >
                <X className="w-6 h-6" />
              </Button>
            </div>
          </div>

          {/* Video Container */}
          <div className="relative group">
            <video
              ref={videoRef}
              className="w-full aspect-video object-cover"
              onTimeUpdate={handleTimeUpdate}
              onLoadedMetadata={handleLoadedMetadata}
              onPlay={() => setIsPlaying(true)}
              onPause={() => setIsPlaying(false)}
              poster="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 1200 675'%3E%3Cdefs%3E%3ClinearGradient id='grad' x1='0%25' y1='0%25' x2='100%25' y2='100%25'%3E%3Cstop offset='0%25' style='stop-color:%23667eea;stop-opacity:1' /%3E%3Cstop offset='100%25' style='stop-color:%235f72bd;stop-opacity:1' /%3E%3C/linearGradient%3E%3C/defs%3E%3Crect width='1200' height='675' fill='url(%23grad)'/%3E%3Ctext x='600' y='337.5' font-family='Arial, sans-serif' font-size='48' fill='white' text-anchor='middle' dy='0.3em'%3EDemo Video%3C/text%3E%3C/svg%3E"
            >
              <source src={videoUrl} type="video/mp4" />
              Your browser does not support the video tag.
            </video>

            {/* Video Overlay Controls */}
            <div className="absolute inset-0 bg-gradient-to-t from-steel-blue-light/70 via-steel-blue-light/20 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300">
              {/* Center Play Button */}
              <div className="absolute inset-0 flex items-center justify-center">
                <Button
                  onClick={togglePlay}
                  size="lg"
                  className="w-24 h-24 rounded-full bg-gradient-to-br from-steel-blue-light/50 to-steel-blue-light/70 hover:from-steel-blue-light/60 hover:to-steel-blue-light/80 backdrop-blur-md transition-all duration-300 hover:scale-110 border-0 shadow-2xl shadow-steel-blue-light/40"
                >
                  {isPlaying ? (
                    <Pause className="w-10 h-10 text-white drop-shadow-lg" />
                  ) : (
                    <Play className="w-10 h-10 text-white ml-1 drop-shadow-lg" />
                  )}
                </Button>
              </div>

              {/* Bottom Controls */}
              <div className="absolute bottom-0 left-0 right-0 p-6">
                {/* Progress Bar */}
                <div 
                  className="w-full h-3 bg-steel-blue-light/20 rounded-full cursor-pointer mb-4 overflow-hidden backdrop-blur-sm border border-steel-blue-light/30 shadow-inner"
                  onClick={handleProgressClick}
                >
                  <div 
                    className="h-full bg-gradient-to-r from-steel-blue-light via-white to-steel-blue-light transition-all duration-150 relative shadow-lg"
                    style={{ width: `${progress}%` }}
                  >
                    <div className="absolute right-0 top-1/2 -translate-y-1/2 w-5 h-5 bg-white rounded-full shadow-xl border-2 border-steel-blue-light/50"></div>
                  </div>
                </div>

                {/* Control Buttons */}
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-4">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={togglePlay}
                      className="text-white hover:bg-steel-blue-light/30 p-3 rounded-xl backdrop-blur-sm border border-steel-blue-light/20 shadow-lg hover:shadow-steel-blue-light/30 transition-all duration-300"
                    >
                      {isPlaying ? <Pause className="w-5 h-5" /> : <Play className="w-5 h-5" />}
                    </Button>
                    
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={toggleMute}
                      className="text-white hover:bg-steel-blue-light/30 p-3 rounded-xl backdrop-blur-sm border border-steel-blue-light/20 shadow-lg hover:shadow-steel-blue-light/30 transition-all duration-300"
                    >
                      {isMuted ? <VolumeX className="w-5 h-5" /> : <Volume2 className="w-5 h-5" />}
                    </Button>

                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={restart}
                      className="text-white hover:bg-steel-blue-light/30 p-3 rounded-xl backdrop-blur-sm border border-steel-blue-light/20 shadow-lg hover:shadow-steel-blue-light/30 transition-all duration-300"
                    >
                      <RotateCcw className="w-5 h-5" />
                    </Button>

                    <span className="text-white text-sm font-mono bg-steel-blue-light/20 px-3 py-1 rounded-lg backdrop-blur-sm border border-steel-blue-light/30 shadow-lg">
                      {formatTime(currentTime)} / {formatTime(duration)}
                    </span>
                  </div>

                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={fullscreen}
                    className="text-white hover:bg-steel-blue-light/30 p-3 rounded-xl backdrop-blur-sm border border-steel-blue-light/20 shadow-lg hover:shadow-steel-blue-light/30 transition-all duration-300"
                  >
                    <Maximize className="w-5 h-5" />
                  </Button>
                </div>
              </div>
            </div>
          </div>

          {/* Footer */}
          <div className="p-6 bg-gradient-to-r from-transparent via-steel-blue-light/20 to-steel-blue-light/40 backdrop-blur-sm">
            <div className="flex items-center justify-between">
              <p className="text-white/90 text-sm font-medium drop-shadow-sm">
                Experience the future of digital innovation
              </p>
              <div className="flex gap-3">
                <div className="w-3 h-3 bg-gradient-to-br from-steel-blue-light to-white rounded-full animate-pulse shadow-lg shadow-steel-blue-light/50"></div>
                <div className="w-3 h-3 bg-gradient-to-br from-white to-steel-blue-light rounded-full animate-pulse delay-75 shadow-lg shadow-steel-blue-light/50"></div>
                <div className="w-3 h-3 bg-gradient-to-br from-steel-blue-light/80 to-steel-blue-light rounded-full animate-pulse delay-150 shadow-lg shadow-steel-blue-light/50"></div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

const Projects = () => {
  const navigate = useNavigate();
  const [videoModal, setVideoModal] = useState({
    isOpen: false,
    videoUrl: '',
    title: '',
    category: ''
  });
  
  const handleContactClick = () => {
    navigate('/contact');
  };

  const openVideoModal = (project) => {
    // Demo video URLs - replace with actual project demo videos
    const demoVideos = {
      "Butchery System": "https://pop-in.co.ke/mathenoid%20ict%20lab/videos/butcherySystem.mp4",
      "Songbook": "https://pop-in.co.ke/mathenoid%20ict%20lab/videos/songbook.mp4",
      "Architectural firm website": "https://pop-in.co.ke/mathenoid%20ict%20lab/videos/arch.mp4"
    };

    setVideoModal({
      isOpen: true,
      videoUrl: demoVideos[project.title] || demoVideos["FinanceFlow Pro"],
      title: project.title,
      category: project.category
    });
  };

  const closeVideoModal = () => {
    setVideoModal({
      isOpen: false,
      videoUrl: '',
      title: '',
      category: ''
    });
  };

  const featuredProjects = [
    {
      title: "Butchery System",
      category: "POS",
      description: "A comprehensive Butchery management platform with insights and real-time analytics.",
      image: "bg-gradient-to-br from-emerald-500 to-teal-600",
      technologies: ["React", "Node.js", "MySQl"],
      metrics: { users: "3+", growth: "150%", rating: "4.9/5" },
      year: "2025",
      status: "Live"
    },
    {
      title: "Songbook",
      category: "Church Systems",
      description: "Interactive songbook app for churches with offline mode.",
      image: "bg-gradient-to-br from-purple-500 to-indigo-600",
      technologies: ["react native", "express js", "MySQL", "SQlite"],
      metrics: { users: "80+", growth: "200%", rating: "4.8/5" },
      year: "2024",
      status: "Live"
    },
    {
      title: "Architectural firm website",
      category: "Web Systems",
      description: "Web based platform connecting clients with architectural firm through showcasing projects and elaborate reach out channels",
      image: "bg-gradient-to-br from-blue-500 to-cyan-600",
      technologies: ["React", "MySQL", "Express js", "Security Compliance"],
      metrics: { users: "1k+", growth: "120%", rating: "4.9/5" },
      year: "2024",
      status: "Live"
    }
  ];

  const otherProjects = [
    {
      title: "E-Commerce Revolution",
      description: "Modern marketplace with AI recommendations",
      tech: ["react.js", "expres js", "mongoDB", "Daraja"],
      year: "2023"
    },
    {
      title: "Social Group management System",
      description: "All round welfare platform",
      tech: ["react", "express", "MySQL", "Firebase", "Daraja"],
      year: "2025"
    },
    {
      title: "Logistics System",
      description: "Secure digital logistics platform",
      tech: ["react", "express.js", "mongoDB","Daraja", "..."],
      year: "2025"
    },
    {
      title: "LMS",
      description: "Virtual classromm tool",
      tech: ["React", "MySQL", "express","Daraja", "..."],
      year: "2025"
    }
  ];

  const achievements = [
    { icon: Award, value: "50+", label: "Projects Completed" },
    { icon: Users, value: "30+", label: "Active Users" },
    { icon: TrendingUp, value: "99%", label: "Client Satisfaction" },
    { icon: Sparkles, value: "3+", label: "Awards Won" }
  ];

  return (
    <div className="min-h-screen">
      <Header />
      
      {/* Hero Section */}
      <section className="pt-32 pb-16 px-4 sm:px-6 lg:px-8">
        <div className="container mx-auto">
          <div className="max-w-4xl mx-auto text-center mb-16">
            <Badge variant="outline" className="mb-6 text-primary border-primary/20 font-tech">
              Our Portfolio
            </Badge>
            <h1 className="font-space-age text-5xl md:text-7xl font-bold mb-6 bg-gradient-to-r from-primary via-purple-500 to-steel-blue-light bg-clip-text text-transparent animate-hologram">
              Projects That Inspire
            </h1>
            <p className="text-xl text-muted-foreground leading-relaxed font-cyber">
              Discover the innovative solutions we've crafted for forward-thinking clients across various industries.
            </p>
          </div>

          {/* Achievements Grid */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mb-16">
            {achievements.map((achievement, index) => {
              const Icon = achievement.icon;
              return (
                <Card key={index} className="p-6 text-center border-0 bg-gradient-to-b from-background/80 to-muted/20 hover:scale-105 transition-transform duration-300">
                  <Icon className="w-8 h-8 mx-auto mb-3 text-primary" />
                  <div className="text-2xl font-bold text-foreground mb-1">{achievement.value}</div>
                  <div className="text-sm text-muted-foreground">{achievement.label}</div>
                </Card>
              );
            })}
          </div>
        </div>
      </section>

      {/* Featured Projects */}
      <section className="py-16 px-4 sm:px-6 lg:px-8">
        <div className="container mx-auto">
          <h2 className="text-3xl md:text-4xl font-bold text-center mb-16 text-foreground">
            Featured Projects
          </h2>
          
          <div className="space-y-16">
            {featuredProjects.map((project, index) => (
              <Card key={index} className="overflow-hidden border-0 shadow-2xl bg-gradient-to-r from-background/50 to-background/80 backdrop-blur-sm">
                <div className={`grid grid-cols-1 lg:grid-cols-2 min-h-[500px]`}>
                  {/* Project Image/Visual */}
                  <div className={`${project.image} flex items-center justify-center relative overflow-hidden ${index % 2 === 1 ? 'lg:order-2' : ''}`}>
                    <div className="absolute inset-0 bg-black/20"></div>
                    <div className="relative z-10 text-center text-white p-8">
                      <Badge className="mb-4 bg-white/20 text-white border-white/30">
                        {project.category}
                      </Badge>
                      <h3 className="text-3xl font-bold mb-4">{project.title}</h3>
                      <div className="flex justify-center space-x-4">
                        <Button 
                          variant="outline" 
                          size="sm" 
                          className="bg-white/20 border-white/30 text-white hover:bg-white/30 transition-all duration-300 hover:scale-105"
                          onClick={() => openVideoModal(project)}
                        >
                          <Eye className="w-4 h-4 mr-2" />
                          View Demo
                        </Button>
                        <Button 
                          variant="outline" 
                          onClick={handleContactClick}
                          size="sm" 
                          className="bg-white/20 border-white/30 text-white hover:bg-white/30 transition-all duration-300 hover:scale-105"
                        >
                          <MessageCircle className="w-4 h-4 mr-2" />
                          Inquiry
                        </Button>
                      </div>
                    </div>
                  </div>

                  {/* Project Details */}
                  <div className="p-8 lg:p-12 flex flex-col justify-center">
                    <div className="flex items-center justify-between mb-4">
                      <Badge variant="secondary">{project.year}</Badge>
                      <Badge className="bg-green-500/20 text-green-600 border-green-500/30">
                        {project.status}
                      </Badge>
                    </div>
                    
                    <h3 className="text-2xl md:text-3xl font-bold mb-4 text-foreground">
                      {project.title}
                    </h3>
                    
                    <p className="text-lg text-muted-foreground mb-6 leading-relaxed">
                      {project.description}
                    </p>

                    {/* Technologies */}
                    <div className="mb-6">
                      <h4 className="text-sm font-semibold text-foreground mb-3">Technologies Used</h4>
                      <div className="flex flex-wrap gap-2">
                        {project.technologies.map((tech, idx) => (
                          <Badge key={idx} variant="outline" className="text-primary border-primary/20">
                            {tech}
                          </Badge>
                        ))}
                      </div>
                    </div>

                    {/* Metrics */}
                    <div className="grid grid-cols-3 gap-4">
                      <div>
                        <div className="text-xl font-bold text-primary">{project.metrics.users}</div>
                        <div className="text-xs text-muted-foreground">Users</div>
                      </div>
                      <div>
                        <div className="text-xl font-bold text-primary">{project.metrics.growth}</div>
                        <div className="text-xs text-muted-foreground">Growth</div>
                      </div>
                      <div>
                        <div className="text-xl font-bold text-primary">{project.metrics.rating}</div>
                        <div className="text-xs text-muted-foreground">Rating</div>
                      </div>
                    </div>
                  </div>
                </div>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Other Projects */}
      <section className="py-16 px-4 sm:px-6 lg:px-8 bg-gradient-to-b from-background to-muted/20">
        <div className="container mx-auto">
          <h2 className="text-3xl md:text-4xl font-bold text-center mb-16 text-foreground">
            More Projects
          </h2>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {otherProjects.map((project, index) => (
              <Card key={index} className="p-6 hover:shadow-xl transition-all duration-300 border-0 bg-background/80 backdrop-blur-sm group hover:scale-105">
                <div className="flex justify-between items-start mb-4">
                  <h3 className="text-xl font-bold text-foreground group-hover:text-primary transition-colors">
                    {project.title}
                  </h3>
                  <Badge variant="outline" className="text-xs">
                    {project.year}
                  </Badge>
                </div>
                
                <p className="text-muted-foreground mb-4">{project.description}</p>
                
                <div className="flex flex-wrap gap-2 mb-4">
                  {project.tech.map((tech, idx) => (
                    <Badge key={idx} variant="secondary" className="text-xs">
                      {tech}
                    </Badge>
                  ))}
                </div>
                
                <Button variant="ghost" size="sm" onClick={handleContactClick} className="w-full group-hover:bg-slate-900">
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Inquire More
                </Button>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Video Modal */}
      <VideoModal
        isOpen={videoModal.isOpen}
        onClose={closeVideoModal}
        videoUrl={videoModal.videoUrl}
        title={videoModal.title}
        category={videoModal.category}
      />

      <Footer />
    </div>
  );
};

export default Projects;